# Troubleshooting Guide

Complete troubleshooting guide for Kleopatra on macOS.

## 🚨 Common Issues

### File Operations Don't Work

#### Problem: Sign/Encrypt buttons do nothing
**Cause**: macOS sandboxing restrictions
**Solution**: ✅ **Use drag-and-drop method instead**

1. Drag files directly onto Kleopatra window
2. Select "Sign/Encrypt..." from popup menu
3. This method works perfectly!

#### Problem: File menu operations fail silently
**Cause**: Same sandboxing issue
**Solution**: Always use drag-and-drop for file operations

### Drag & Drop Issues

#### Problem: Drag & drop doesn't work at all
**Solutions**:

1. **Grant Full Disk Access**:
   - System Settings → Privacy & Security → Full Disk Access
   - Add: `/opt/homebrew/Applications/kleopatra.app/Contents/MacOS/kleopatra`
   - Restart Kleopatra

2. **Launch from Terminal**:
   ```bash
   open $(brew --prefix)/Applications/kleopatra.app
   ```

3. **Try different file locations**:
   - Move files to Desktop or Documents
   - Avoid system folders or external drives

#### Problem: "Permission denied" when dragging files
**Solution**: Check file permissions and ownership
```bash
# Check file permissions
ls -la yourfile.txt

# Fix permissions if needed
chmod 644 yourfile.txt
```

## 🔧 Service Issues

### DBus Problems

#### Problem: "DBus connection failed" errors
**Solutions**:

1. **Start DBus service**:
   ```bash
   brew services start dbus
   ```

2. **Restart DBus**:
   ```bash
   brew services restart dbus
   ```

3. **Check DBus status**:
   ```bash
   brew services list | grep dbus
   # Should show "started"
   ```

#### Problem: DBus won't start
**Solution**: Reset DBus configuration
```bash
# Stop DBus
brew services stop dbus

# Remove configuration
rm -rf ~/Library/LaunchAgents/org.freedesktop.dbus-session.plist

# Reinstall DBus
brew reinstall dbus

# Restart
brew services start dbus
```

### GPG Agent Issues

#### Problem: "GPG agent not running" errors
**Solutions**:

1. **Restart GPG agent**:
   ```bash
   killall -9 gpg-agent
   gpg-agent --daemon
   ```

2. **Check GPG agent status**:
   ```bash
   gpg --version
   gpgconf --list-components
   ```

3. **Fix pinentry configuration**:
   ```bash
   echo "pinentry-program $(brew --prefix)/bin/pinentry-mac" >> ~/.gnupg/gpg-agent.conf
   killall -9 gpg-agent
   ```

#### Problem: Wrong pinentry program
**Solution**: Configure pinentry-mac
```bash
# Check current pinentry
gpgconf --list-options gpg-agent | grep pinentry-program

# Set correct pinentry
echo "pinentry-program $(brew --prefix)/bin/pinentry-mac" > ~/.gnupg/gpg-agent.conf

# Restart agent
gpgconf --kill gpg-agent
```

## 🗝️ Key Management Problems

### Import/Export Issues

#### Problem: Can't import keys via drag & drop
**Solutions**:

1. **Use File menu**: File → Import Certificates (this menu works!)
2. **Copy-paste key text**: Select All → Copy from key file, paste in import dialog
3. **Use GPG command line**:
   ```bash
   gpg --import keyfile.asc
   ```

#### Problem: Exported keys are corrupted
**Solution**: Use ASCII armor format
1. Right-click key → Export Certificate
2. Choose ".asc" extension (ASCII format)
3. Verify exported key:
   ```bash
   gpg --show-keys exported-key.asc
   ```

### Key Generation Issues

#### Problem: Key generation fails or crashes
**Solutions**:

1. **Check entropy**:
   ```bash
   # Generate some entropy
   find / -name "*.log" -type f -exec wc -l {} \; > /dev/null &
   ```

2. **Use command line**:
   ```bash
   gpg --full-gen-key
   ```

3. **Restart Kleopatra**:
   ```bash
   killall kleopatra
   kleopatra
   ```

## 🖥️ Application Issues

### Launch Problems

#### Problem: Kleopatra won't start
**Solutions**:

1. **Check dependencies**:
   ```bash
   brew list | grep kf6-
   # Should show many kf6-* packages
   ```

2. **Reinstall missing dependencies**:
   ```bash
   brew reinstall kf6-kcoreaddons kf6-kconfig kf6-ki18n
   ```

3. **Launch with verbose output**:
   ```bash
   $(brew --prefix)/bin/kleopatra --version
   ```

#### Problem: "dylib not found" errors
**Solutions**:

1. **Check library paths**:
   ```bash
   otool -L $(brew --prefix)/bin/kleopatra
   ```

2. **Fix libassuan symlink**:
   ```bash
   cd $(brew --prefix)/lib
   ln -sf libassuan.*.dylib libassuan.0.dylib
   ```

3. **Reinstall Kleopatra**:
   ```bash
   brew uninstall kleopatra
   brew install kleopatra
   ```

### Performance Issues

#### Problem: Kleopatra is slow or freezes
**Solutions**:

1. **Restart with fresh profile**:
   ```bash
   mv ~/Library/Preferences/org.kde.kleopatra.plist ~/Library/Preferences/org.kde.kleopatra.plist.bak
   kleopatra
   ```

2. **Clear certificate cache**:
   ```bash
   rm -rf ~/.gnupg/trustdb.gpg
   gpg --check-trustdb
   ```

3. **Reduce certificate list**:
   - Delete unused certificates
   - Refresh certificate list (Cmd+R)

## 🌐 Network Issues

### Key Server Problems

#### Problem: Can't connect to key servers
**Solutions**:

1. **Test network connectivity**:
   ```bash
   ping keys.openpgp.org
   ```

2. **Configure key servers**:
   ```bash
   echo "keyserver hkps://keys.openpgp.org" >> ~/.gnupg/gpg.conf
   ```

3. **Use alternative key servers**:
   - hkps://keyserver.ubuntu.com
   - hkps://pgp.mit.edu

#### Problem: Key server timeouts
**Solutions**:

1. **Increase timeout**:
   ```bash
   echo "keyserver-options timeout=30" >> ~/.gnupg/gpg.conf
   ```

2. **Use web key directory**:
   ```bash
   gpg --auto-key-locate wkd --locate-keys user@example.com
   ```

## 🔍 Diagnostic Commands

### System Information
```bash
# macOS version
sw_vers

# Homebrew info
brew --version
brew --prefix

# Kleopatra version
kleopatra --version

# GPG version
gpg --version
```

### Check Dependencies
```bash
# List KF6 packages
brew list | grep kf6- | sort

# Check Qt6
brew list qt@6

# Check GPG components
gpgconf --list-components
```

### Service Status
```bash
# DBus status
brew services list | grep dbus

# GPG agent status  
ps aux | grep gpg-agent

# Library dependencies
otool -L $(brew --prefix)/bin/kleopatra | head -20
```

### Log Files
```bash
# Kleopatra logs
tail -f ~/Library/Logs/kleopatra.log

# System logs
log stream --predicate 'process CONTAINS "kleopatra"'

# GPG logs
tail -f ~/.gnupg/gpg-agent.log
```

## 🚑 Emergency Fixes

### Complete Reset
```bash
# Stop all services
brew services stop dbus
killall -9 gpg-agent
killall kleopatra

# Backup and reset configuration
mv ~/.gnupg ~/.gnupg.backup
mv ~/Library/Preferences/org.kde.kleopatra.plist ~/Library/Preferences/org.kde.kleopatra.plist.bak

# Reinstall Kleopatra
brew uninstall kleopatra
brew install kleopatra

# Restart services
brew services start dbus
kleopatra
```

### Restore from Backup
```bash
# Stop services
brew services stop dbus
killall -9 gpg-agent

# Restore configuration
mv ~/.gnupg.backup ~/.gnupg
mv ~/Library/Preferences/org.kde.kleopatra.plist.bak ~/Library/Preferences/org.kde.kleopatra.plist

# Restart
brew services start dbus
kleopatra
```

## 🆘 Getting Help

### Before Reporting Issues

1. **Check this troubleshooting guide** first
2. **Search existing issues**: [GitHub Issues](https://github.com/yourusername/homebrew-kleopatra-modern/issues)
3. **Try the emergency reset** procedures above
4. **Gather diagnostic information** using commands above

### Reporting Bugs

Include this information:
```bash
# System info
sw_vers
brew --version
kleopatra --version

# Dependencies
brew list | grep -E "(kf6-|qt@6|gpg)" | sort

# Services  
brew services list | grep -E "(dbus|gpg)"

# Recent logs
tail -20 ~/Library/Logs/kleopatra.log
```

### Community Support

- **GitHub Discussions**: General questions and tips
- **KDE Forums**: Kleopatra-specific questions  
- **Homebrew Issues**: Formula-related problems
- **GPG Mailing Lists**: GPG/cryptography questions

## ✅ Prevention Tips

1. **Regular Updates**:
   ```bash
   brew update && brew upgrade
   ```

2. **Backup Keys Regularly**:
   ```bash
   gpg --export-secret-keys > my-keys-backup.asc
   ```

3. **Monitor Services**:
   ```bash
   brew services list | grep -E "(dbus|gpg)"
   ```

4. **Use Drag & Drop**: Avoid file menu operations that don't work

5. **Launch from Terminal**: For better file system access