# Development Guide

Guide for developers who want to contribute to the Kleopatra Homebrew formula or understand the build process.

## 🏗️ Architecture Overview

### Components
- **Kleopatra**: Qt6/KF6-based GUI application
- **Homebrew Formula**: Ruby-based package definition
- **Dependencies**: 40+ KDE Frameworks 6 libraries
- **GPG Backend**: GnuPG, GPGME, libassuan integration
- **macOS Integration**: DBus, app bundle, sandboxing workarounds

### Build Process
1. **Download**: Kleopatra 24.02.2 source from KDE
2. **Dependencies**: Install Qt6 and KDE Frameworks 6
3. **Configure**: CMake with macOS-specific flags
4. **Build**: Ninja-based compilation
5. **Install**: Create .app bundle and fix library paths

## 🛠️ Development Setup

### Prerequisites
```bash
# Essential tools
brew install cmake ninja pkg-config git

# Build dependencies  
brew install extra-cmake-modules gettext

# Qt6 and core frameworks
brew install qt@6 kf6-kcoreaddons kf6-kconfig kf6-ki18n
```

### Clone and Setup
```bash
# Clone this repository
git clone https://github.com/yourusername/homebrew-kleopatra-modern.git
cd homebrew-kleopatra-modern

# Install in development mode
brew install --build-from-source --verbose ./Formula/kleopatra.rb
```

## 📋 Formula Development

### Ruby Formula Structure
```ruby
class Kleopatra < Formula
  # Metadata
  desc, homepage, url, sha256, license
  
  # Dependencies  
  depends_on "build-tools" => :build
  depends_on "runtime-libraries"
  depends_on "kf6-*" # KDE Frameworks 6
  
  # Build process
  def install
    # CMake configuration
    # Build with make/ninja
    # Install and fix paths
  end
  
  # Post-installation setup
  def post_install
    # Configure services
    # Set up GPG agent
  end
  
  # Installation notes
  def caveats; end
  
  # Verification tests
  def test; end
end
```

### Key Formula Components

#### Dependencies Management
```ruby
# Build-only dependencies
depends_on "cmake" => :build
depends_on "ninja" => :build

# Runtime dependencies with specific versions
depends_on "qt@6"
depends_on "kf6-kcoreaddons"
```

#### CMake Configuration
```ruby
args = [
  "-DCMAKE_INSTALL_PREFIX=#{prefix}",
  "-DCMAKE_BUILD_TYPE=Release", 
  "-DBUILD_TESTING=OFF",
  "-DQT_MAJOR_VERSION=6",
  "-DCMAKE_FIND_FRAMEWORK=LAST"
]
```

#### macOS-Specific Fixes
```ruby
# Fix library linking issues
MachO::Tools.change_install_name(
  prefix/"bin/kleopatra",
  old_path,
  new_path
)
```

### Testing the Formula

#### Local Testing
```bash
# Syntax check
brew ruby -c Formula/kleopatra.rb

# Install test
brew install --build-from-source ./Formula/kleopatra.rb

# Run tests
brew test kleopatra

# Uninstall
brew uninstall kleopatra
```

#### Comprehensive Testing
```bash
# Test on clean system
brew uninstall --ignore-dependencies kleopatra
brew uninstall --ignore-dependencies $(brew deps kleopatra)

# Reinstall from scratch
brew install ./Formula/kleopatra.rb

# Verify functionality
kleopatra --version
echo "test" > test.txt
# Drag test.txt onto Kleopatra and verify drag-drop works
```

## 🔧 Build Process Details

### Manual Build (for debugging)
```bash
# Download source
curl -L -O https://download.kde.org/stable/release-service/24.02.2/src/kleopatra-24.02.2.tar.xz
tar xf kleopatra-24.02.2.tar.xz
cd kleopatra-24.02.2

# Configure build
cmake . \
  -DCMAKE_INSTALL_PREFIX=/opt/homebrew \
  -DCMAKE_BUILD_TYPE=Release \
  -DBUILD_TESTING=OFF \
  -DQT_MAJOR_VERSION=6 \
  -G Ninja

# Build
ninja -j8

# Install  
ninja install
```

### Common Build Issues

#### Missing KF6 Dependencies
**Problem**: CMake can't find KDE Frameworks
**Solution**: Install missing packages
```bash
# Check what's installed
brew list | grep kf6-

# Install missing frameworks
brew install kf6-karchive kf6-ki18n kf6-kconfig
```

#### Library Linking Errors
**Problem**: Runtime dylib not found errors
**Solution**: Fix install names
```bash
# Check current paths
otool -L /opt/homebrew/bin/kleopatra

# Fix problematic libraries
install_name_tool -change old_path new_path /opt/homebrew/bin/kleopatra
```

#### Qt6 Detection Issues
**Problem**: CMake can't find Qt6
**Solution**: Set CMAKE_PREFIX_PATH
```bash
export CMAKE_PREFIX_PATH="/opt/homebrew/opt/qt@6:$CMAKE_PREFIX_PATH"
```

## 🧪 Testing Infrastructure

### GitHub Actions Setup
```yaml
name: Test Kleopatra Formula
on: [push, pull_request]

jobs:
  test:
    runs-on: macos-latest
    steps:
    - uses: actions/checkout@v3
    - name: Install formula
      run: brew install --build-from-source ./Formula/kleopatra.rb
    - name: Test installation
      run: |
        kleopatra --version
        brew test kleopatra
```

### Manual Testing Checklist

#### Installation Testing
- [ ] Formula installs without errors
- [ ] All dependencies are satisfied  
- [ ] Binary is created and executable
- [ ] .app bundle is properly formed
- [ ] Services (DBus) start correctly

#### Functionality Testing
- [ ] Kleopatra launches successfully
- [ ] Drag & drop works for file operations
- [ ] Key generation works
- [ ] Key import/export works
- [ ] GPG operations complete successfully

#### Integration Testing
- [ ] Works with existing GPG setup
- [ ] DBus integration functional
- [ ] macOS app bundle opens from Applications
- [ ] Command line `kleopatra` command works

## 📦 Release Process

### Version Updates
1. **Check KDE releases**: https://kde.org/announcements/
2. **Update formula**:
   ```ruby
   url "https://download.kde.org/stable/release-service/NEW_VERSION/src/kleopatra-NEW_VERSION.tar.xz"
   sha256 "new-hash-here"
   ```
3. **Calculate new SHA256**:
   ```bash
   curl -L URL | shasum -a 256
   ```

### Testing New Versions
```bash
# Update formula with new version
vim Formula/kleopatra.rb

# Test build
brew install --build-from-source ./Formula/kleopatra.rb

# Verify functionality
kleopatra --version
# Test drag-drop operations
# Test key management

# Check for regressions
brew test kleopatra
```

### Publishing Updates
1. **Test thoroughly** on clean macOS system
2. **Update documentation** if needed
3. **Create pull request** with changes
4. **Tag release** after merge

## 🐛 Debugging

### Build Debugging
```bash
# Verbose build output
brew install --build-from-source --verbose ./Formula/kleopatra.rb

# CMake debugging
cmake . -DCMAKE_VERBOSE_MAKEFILE=ON

# Library dependency debugging
otool -L $(brew --prefix)/bin/kleopatra
```

### Runtime Debugging
```bash
# Launch with debug output
DYLD_PRINT_LIBRARIES=1 kleopatra

# GPG debugging
gpg --debug-level guru --version

# DBus debugging  
dbus-monitor --session
```

### Log Analysis
```bash
# Kleopatra logs
tail -f ~/Library/Logs/kleopatra.log

# System logs
log stream --predicate 'process CONTAINS "kleopatra"'

# Homebrew logs
brew config
brew doctor
```

## 🤝 Contributing

### Code Style
- Follow Homebrew formula conventions
- Use 2-space indentation for Ruby
- Include descriptive comments
- Test changes thoroughly

### Documentation Updates
- Update README.md for user-facing changes
- Update this development guide for build changes
- Include migration notes for breaking changes

### Pull Request Process
1. **Fork repository**
2. **Create feature branch**: `git checkout -b feature/description`
3. **Make changes** with tests
4. **Update documentation**
5. **Test thoroughly**:
   ```bash
   brew install --build-from-source ./Formula/kleopatra.rb
   brew test kleopatra
   ```
6. **Submit pull request** with detailed description

### Issue Reporting
Include:
- macOS version (`sw_vers`)
- Homebrew version (`brew --version`)
- Build logs (if build failure)
- Runtime error messages
- Steps to reproduce

## 🔬 Advanced Topics

### Cross-Platform Considerations
- macOS sandboxing affects file operations
- Qt6/KF6 frameworks behave differently on macOS
- DBus implementation varies from Linux
- Library linking requires macOS-specific fixes

### Performance Optimization
- Use Ninja for faster builds
- Parallel compilation with `-j$(nproc)`
- Bottle creation for pre-built binaries
- Dependency optimization

### Security Considerations
- GPG key handling and storage
- macOS keychain integration possibilities  
- Sandboxing implications
- Code signing for distribution

## 📚 Resources

### KDE Development
- [KDE Developer Portal](https://develop.kde.org/)
- [Kleopatra Git Repository](https://invent.kde.org/pim/kleopatra)
- [KDE Frameworks Documentation](https://api.kde.org/frameworks/)

### Homebrew Development  
- [Formula Cookbook](https://docs.brew.sh/Formula-Cookbook)
- [Homebrew Best Practices](https://docs.brew.sh/Acceptable-Formulae)
- [Testing Guide](https://docs.brew.sh/Formula-Cookbook#add-a-test-to-the-formula)

### macOS Development
- [Apple Developer Documentation](https://developer.apple.com/documentation/)
- [macOS App Sandboxing](https://developer.apple.com/documentation/security/app_sandbox)
- [macOS Library Management](https://developer.apple.com/library/archive/documentation/DeveloperTools/Conceptual/DynamicLibraries/)