class Kleopatra < Formula
  desc "Modern Kleopatra - Certificate manager and GUI for OpenPGP and CMS cryptography"
  homepage "https://apps.kde.org/kleopatra/"
  url "https://download.kde.org/stable/release-service/24.02.2/src/kleopatra-24.02.2.tar.xz"
  sha256 "your-sha256-hash-here"
  license all_of: ["GPL-2.0-only", "GPL-3.0-only", "LGPL-2.1-only", "LGPL-3.0-only"]
  head "https://invent.kde.org/pim/kleopatra.git", branch: "master"

  depends_on "cmake" => :build
  depends_on "extra-cmake-modules" => :build
  depends_on "gettext" => :build
  depends_on "ninja" => :build
  depends_on "pkg-config" => :build

  # Runtime dependencies
  depends_on "boost"
  depends_on "dbus"
  depends_on "gnupg"
  depends_on "gpgme"
  depends_on "libassuan"
  depends_on "libgpg-error"
  depends_on "pinentry-mac"
  depends_on "qt@6"

  # KDE Frameworks 6 dependencies
  depends_on "kf6-karchive"
  depends_on "kf6-kauth"
  depends_on "kf6-kcodecs"
  depends_on "kf6-kcompletion"
  depends_on "kf6-kconfig"
  depends_on "kf6-kconfigwidgets"
  depends_on "kf6-kcoreaddons"
  depends_on "kf6-kcrash"
  depends_on "kf6-kdbusaddons"
  depends_on "kf6-ki18n"
  depends_on "kf6-kiconthemes"
  depends_on "kf6-kitemmodels"
  depends_on "kf6-kitemviews"
  depends_on "kf6-kjobwidgets"
  depends_on "kf6-knotifications"
  depends_on "kf6-kservice"
  depends_on "kf6-kstatusnotifieritem"
  depends_on "kf6-ktextwidgets"
  depends_on "kf6-kwidgetsaddons"
  depends_on "kf6-kwindowsystem"
  depends_on "kf6-kxmlgui"

  def install
    # Set up build environment
    ENV["CMAKE_PREFIX_PATH"] = [
      Formula["qt@6"].opt_prefix,
      prefix
    ].join(";")

    args = [
      "-DCMAKE_INSTALL_PREFIX=#{prefix}",
      "-DCMAKE_BUILD_TYPE=Release",
      "-DBUILD_TESTING=OFF",
      "-DDISABLE_KWATCHGNUPG=OFF",
      "-DQT_MAJOR_VERSION=6",
      "-DCMAKE_FIND_FRAMEWORK=LAST",
      "-DCMAKE_VERBOSE_MAKEFILE=ON",
      "-Wno-dev"
    ]

    system "cmake", ".", *args
    system "make", "-j#{ENV.make_jobs}"
    system "make", "install"

    # Create proper macOS app bundle
    app_bundle = prefix/"Applications/KDE/kleopatra.app"
    if app_bundle.exist?
      (prefix/"Applications").install app_bundle
    end

    # Fix library paths for macOS
    MachO::Tools.change_install_name(
      prefix/"bin/kleopatra",
      "/opt/homebrew/opt/libassuan/lib/libassuan.0.dylib",
      Formula["libassuan"].opt_lib/"libassuan.#{Formula["libassuan"].version.major}.dylib"
    )
  end

  def post_install
    # Ensure dbus is running
    system "brew", "services", "start", "dbus" unless quiet_system("brew", "services", "list", "dbus") && $?.success?

    # Set up pinentry configuration
    gnupg_dir = Pathname.new(Dir.home)/".gnupg"
    gnupg_dir.mkpath
    gpg_agent_conf = gnupg_dir/"gpg-agent.conf"
    
    unless gpg_agent_conf.exist? && gpg_agent_conf.read.include?("pinentry-program")
      gpg_agent_conf.write("pinentry-program #{Formula["pinentry-mac"].opt_bin}/pinentry-mac\n", 
                          perm: 0o600)
      system "killall", "-9", "gpg-agent" if system "pgrep", "gpg-agent"
    end

    # Create symlink for easy access
    (HOMEBREW_PREFIX/"bin").install_symlink prefix/"bin/kleopatra" => "kleopatra"
  end

  def caveats
    <<~EOS
      Kleopatra has been installed successfully!

      🔧 IMPORTANT SETUP NOTES:

      1. Launch Kleopatra:
         • Applications folder: Look for Kleopatra.app
         • Command line: kleopatra
         • Terminal launch (recommended): open #{prefix}/Applications/kleopatra.app

      2. File Operations - IMPORTANT:
         ❌ The Sign/Encrypt BUTTONS and FILE MENU don't work due to macOS sandboxing
         ✅ USE DRAG & DROP instead:
           - Drag files directly onto the Kleopatra window
           - Select "Sign/Encrypt..." or "Decrypt/Verify..." from the popup menu
           - This method works perfectly!

      3. Grant Full Disk Access (if needed):
         • System Settings → Privacy & Security → Full Disk Access
         • Add: #{prefix}/Applications/kleopatra.app/Contents/MacOS/kleopatra

      4. Services:
         • DBus: #{quiet_system("brew", "services", "list", "dbus") ? "✅ Running" : "⚠️  Start with: brew services start dbus"}
         • GPG Agent: Configured automatically

      For detailed documentation: https://github.com/yourusername/homebrew-kleopatra-modern

      The drag-and-drop method is actually faster than file dialogs once you get used to it!
    EOS
  end

  test do
    # Test that the binary exists and shows version
    assert_match version.to_s, shell_output("#{bin}/kleopatra --version", 1)
    
    # Test that required libraries can be loaded
    system bin/"kleopatra", "--help"
  end
end